/* Copyright (C) 2015-2018 RealVNC Ltd.  All Rights Reserved.
 */

#include <vnccommon/ConditionVariable.h>
#include <vnccommon/StringUtils.h>

#include <sys/time.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>

#include "ConditionAttribute.h"

using namespace vnccommon;

ConditionVariable::ConditionVariable()
{
    mAttr.reset(new ConditionAttribute());

    const int cond_result = pthread_cond_init(
            &mCondition,
            mAttr->pthreadCondAttr());

    if(cond_result != 0)
    {
        std::string message = vnccommon::StringUtils::format(
                "pthread_cond_init failed. %s", strerror(cond_result));
        throw ConditionVariableException(message);
    }
}

void ConditionVariable::signal()
{
    const int result = pthread_cond_signal(&mCondition);
    if(result != 0)
    {
        std::string message = vnccommon::StringUtils::format(
                "pthread_cond_signal failed. %s", strerror(result));
        throw ConditionVariableException(message);
    }
}

void ConditionVariable::wait(Mutex::Locker& mutex)
{
    const int result = pthread_cond_wait(&mCondition, &mutex.mMutex.internalGet());
    if(result != 0)
    {
        std::string message = vnccommon::StringUtils::format(
                "pthread_cond_wait failed. %s", strerror(result));
        throw ConditionVariableException(message);
    }
}

bool ConditionVariable::waitWithTimeout(Mutex::Locker& mutex, Duration timeoutMs)
{
    if(timeoutMs.getMs() <= 0)
    {
        return false;
    }

    timespec absTimeout;

#ifndef __APPLE__
    if (mAttr->isMonotonicClockAvailable())
    {
        const int clock_error = clock_gettime(CLOCK_MONOTONIC, &absTimeout);
        if (clock_error != 0)
        {
            std::string message = vnccommon::StringUtils::format(
                    "clock_gettime failed. %s", strerror(clock_error));
            throw ConditionVariableException(message);
        }
        // At this point, absTimeout is set to the current monotonic time
    }
    else
#endif
    {
        struct timeval now;
        gettimeofday(&now, NULL);

        absTimeout.tv_sec = now.tv_sec;
        absTimeout.tv_nsec = now.tv_usec * 1000;
        // At this point, absTimeout is set to the current system time
    }

    // Add the timeout to the current time
    absTimeout.tv_sec += timeoutMs.getMs() / 1000;
    absTimeout.tv_nsec += (timeoutMs.getMs() % 1000) * 1000000L;

    absTimeout.tv_sec += absTimeout.tv_nsec / 1000000000L;
    absTimeout.tv_nsec = absTimeout.tv_nsec % 1000000000L;

    int waitResult;

#ifdef NON_STANDARD_PTHREAD
    // Android below 5.0 does not use pthread_cond_timedwait
    waitResult = pthread_cond_timedwait_monotonic(
            &mCondition,
            &mutex.mMutex.internalGet(),
            &absTimeout);
#else
    waitResult = pthread_cond_timedwait(
            &mCondition,
            &mutex.mMutex.internalGet(),
            &absTimeout);
#endif

    switch (waitResult)
    {
        case 0:
            return true;

        case ETIMEDOUT:
            return false;

        default:
            std::string message = vnccommon::StringUtils::format(
                    "pthread_cond_timedwait failed. %s", strerror(waitResult));
            throw ConditionVariableException(message);
    }
}

ConditionVariable::~ConditionVariable()
{
    pthread_cond_destroy(&mCondition);
}

